<?php
/**
 * صفحه مدیریت سفارش‌ها
 * این صفحه امکان مدیریت سفارش‌ها، بررسی وضعیت آنها و تایید یا رد سفارش‌ها را فراهم می‌کند
 */

// فایل‌های مورد نیاز را وارد کنید
require_once '../config/order.php';

// انجام عملیات درخواستی
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // تغییر وضعیت سفارش
    if (isset($_POST['action']) && $_POST['action'] === 'change_status' && isset($_POST['order_id']) && isset($_POST['status'])) {
        $order_id = $_POST['order_id'];
        $status = $_POST['status'];
        $admin_note = $_POST['admin_note'] ?? '';
        
        // بررسی وضعیت فعلی سفارش
        $order = get_order($order_id);
        
        if ($order) {
            if ($status === 'approved' && $order['status'] !== 'approved') {
                // تایید سفارش و اعمال آن به کاربر
                $result = update_order_status($order_id, $status, $admin_note);
                
                if ($result) {
                    // اعمال سفارش به کاربر
                    apply_order_to_user($order_id);
                    $success_message = "سفارش با موفقیت تایید و اعمال شد.";
                } else {
                    $error_message = "خطا در تایید سفارش.";
                }
            } else {
                // سایر تغییرات وضعیت
                $result = update_order_status($order_id, $status, $admin_note);
                
                if ($result) {
                    $status_text = '';
                    switch ($status) {
                        case 'rejected':
                            $status_text = 'رد شده';
                            break;
                        case 'pending':
                            $status_text = 'در انتظار بررسی';
                            break;
                        case 'approved':
                            $status_text = 'تایید شده';
                            break;
                        case 'refunded':
                            $status_text = 'مسترد شده';
                            break;
                        default:
                            $status_text = $status;
                    }
                    $success_message = "وضعیت سفارش به «{$status_text}» تغییر یافت.";
                } else {
                    $error_message = "خطا در تغییر وضعیت سفارش.";
                }
            }
        } else {
            $error_message = "سفارش مورد نظر یافت نشد.";
        }
    }
    
    // حذف سفارش‌ها
    if (isset($_POST['action']) && $_POST['action'] === 'delete' && isset($_POST['order_ids'])) {
        $order_ids = $_POST['order_ids'];
        $deleted_count = 0;
        
        global $pdo;
        foreach ($order_ids as $order_id) {
            try {
                $stmt = $pdo->prepare("DELETE FROM orders WHERE id = ?");
                $stmt->execute([$order_id]);
                if ($stmt->rowCount() > 0) {
                    $deleted_count++;
                }
            } catch (PDOException $e) {
                // خطا در حذف سفارش
            }
        }
        
        $success_message = "{$deleted_count} سفارش با موفقیت حذف شد.";
    }
}

// تنظیمات صفحه‌بندی
$page = isset($_GET['page_num']) ? (int)$_GET['page_num'] : 1;
$per_page = 15;

// فیلترهای جستجو
$search = isset($_GET['search']) ? $_GET['search'] : '';
$filter_status = isset($_GET['status']) ? $_GET['status'] : '';

// دریافت لیست سفارش‌ها
$orders = [];
$total_orders = 0;

global $pdo;

// ساخت کوئری
$sql = "SELECT o.*, u.username FROM orders o
        JOIN users u ON o.user_id = u.id";

$where_clauses = [];
$params = [];

if (!empty($search)) {
    $where_clauses[] = "(u.username LIKE ? OR o.id LIKE ?)";
    $params[] = "%{$search}%";
    $params[] = "%{$search}%";
}

if (!empty($filter_status) && $filter_status !== 'all') {
    $where_clauses[] = "o.status = ?";
    $params[] = $filter_status;
}

if (!empty($where_clauses)) {
    $sql .= " WHERE " . implode(" AND ", $where_clauses);
}

$sql .= " ORDER BY o.created_at DESC LIMIT ? OFFSET ?";
$params[] = $per_page;
$params[] = ($page - 1) * $per_page;

// اجرای کوئری
try {
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $orders = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    // خطا در دریافت سفارش‌ها
}

// دریافت تعداد کل سفارش‌ها برای صفحه‌بندی
$count_sql = "SELECT COUNT(*) FROM orders o JOIN users u ON o.user_id = u.id";
if (!empty($where_clauses)) {
    $count_sql .= " WHERE " . implode(" AND ", $where_clauses);
}

try {
    $count_stmt = $pdo->prepare($count_sql);
    $count_stmt->execute(array_slice($params, 0, -2));
    $total_orders = $count_stmt->fetchColumn();
} catch (PDOException $e) {
    // خطا در شمارش سفارش‌ها
}

$total_pages = ceil($total_orders / $per_page);

// آمار سفارش‌ها
$order_stats = [
    'pending' => 0,
    'approved' => 0,
    'rejected' => 0,
    'refunded' => 0
];

try {
    $stats_sql = "SELECT status, COUNT(*) as count FROM orders GROUP BY status";
    $stats_stmt = $pdo->query($stats_sql);
    
    while ($row = $stats_stmt->fetch(PDO::FETCH_ASSOC)) {
        $order_stats[$row['status']] = $row['count'];
    }
} catch (PDOException $e) {
    // خطا در دریافت آمار
}
?>

<div class="container-fluid">
    <h1 class="h3 mb-4 text-gray-800">مدیریت سفارش‌ها</h1>
    
    <?php if (isset($success_message)): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <?php echo $success_message; ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    <?php endif; ?>
    
    <?php if (isset($error_message)): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <?php echo $error_message; ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    <?php endif; ?>
    
    <!-- کارت‌های آمار -->
    <div class="row mb-4">
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-primary h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-primary text-uppercase mb-1">در انتظار بررسی</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo $order_stats['pending'] ?? 0; ?></div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-clock fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-success h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-success text-uppercase mb-1">تایید شده</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo $order_stats['approved'] ?? 0; ?></div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-check-circle fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-danger h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-danger text-uppercase mb-1">رد شده</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo $order_stats['rejected'] ?? 0; ?></div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-times-circle fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-warning h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-warning text-uppercase mb-1">مسترد شده</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800"><?php echo $order_stats['refunded'] ?? 0; ?></div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-undo fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- فیلتر و جستجو -->
    <div class="card dashboard-card mb-4">
        <div class="card-body">
            <div class="row">
                <div class="col-md-8">
                    <form action="" method="get">
                        <input type="hidden" name="page" value="orders">
                        <div class="input-group">
                            <input type="text" class="form-control" placeholder="جستجو بر اساس شناسه سفارش یا نام کاربری..." name="search" value="<?php echo htmlspecialchars($search); ?>">
                            <select class="form-select" name="status">
                                <option value="" <?php echo $filter_status === '' ? 'selected' : ''; ?>>همه وضعیت‌ها</option>
                                <option value="pending" <?php echo $filter_status === 'pending' ? 'selected' : ''; ?>>در انتظار بررسی</option>
                                <option value="approved" <?php echo $filter_status === 'approved' ? 'selected' : ''; ?>>تایید شده</option>
                                <option value="rejected" <?php echo $filter_status === 'rejected' ? 'selected' : ''; ?>>رد شده</option>
                                <option value="refunded" <?php echo $filter_status === 'refunded' ? 'selected' : ''; ?>>مسترد شده</option>
                            </select>
                            <button class="btn btn-primary" type="submit">
                                <i class="fas fa-search"></i>
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
    
    <!-- جدول سفارش‌ها -->
    <div class="card dashboard-card">
        <div class="card-body">
            <form action="" method="post" id="orderActionForm">
                <div class="table-responsive">
                    <table class="table table-bordered table-hover table-striped">
                        <thead>
                            <tr>
                                <th>
                                    <input type="checkbox" id="selectAll">
                                </th>
                                <th>شناسه</th>
                                <th>کاربر</th>
                                <th>نوع</th>
                                <th>مبلغ</th>
                                <th>تاریخ</th>
                                <th>وضعیت</th>
                                <th>عملیات</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($orders)): ?>
                                <tr>
                                    <td colspan="8" class="text-center">سفارشی یافت نشد.</td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($orders as $order): ?>
                                    <?php
                                    // تعیین نوع سفارش به صورت فارسی
                                    $order_type_text = 'نامشخص';
                                    switch ($order['order_type']) {
                                        case 'new':
                                            $order_type_text = 'جدید';
                                            break;
                                        case 'renewal':
                                            $order_type_text = 'تمدید';
                                            break;
                                        case 'upgrade':
                                            $order_type_text = 'ارتقا';
                                            break;
                                    }
                                    
                                    // تعیین کلاس وضعیت
                                    $status_class = 'secondary';
                                    $status_text = 'نامشخص';
                                    
                                    switch ($order['status']) {
                                        case 'pending':
                                            $status_class = 'primary';
                                            $status_text = 'در انتظار بررسی';
                                            break;
                                        case 'approved':
                                            $status_class = 'success';
                                            $status_text = 'تایید شده';
                                            break;
                                        case 'rejected':
                                            $status_class = 'danger';
                                            $status_text = 'رد شده';
                                            break;
                                        case 'refunded':
                                            $status_class = 'warning';
                                            $status_text = 'مسترد شده';
                                            break;
                                    }
                                    ?>
                                    <tr>
                                        <td>
                                            <input type="checkbox" name="order_ids[]" value="<?php echo $order['id']; ?>" class="item-checkbox">
                                        </td>
                                        <td>#<?php echo $order['id']; ?></td>
                                        <td>
                                            <a href="?page=user_edit&id=<?php echo $order['user_id']; ?>">
                                                <?php echo htmlspecialchars($order['username']); ?>
                                            </a>
                                        </td>
                                        <td><?php echo $order_type_text; ?></td>
                                        <td><?php echo number_format($order['amount']); ?> تومان</td>
                                        <td><?php echo jdate('Y/m/d H:i', strtotime($order['created_at'])); ?></td>
                                        <td>
                                            <span class="badge bg-<?php echo $status_class; ?>">
                                                <?php echo $status_text; ?>
                                            </span>
                                        </td>
                                        <td>
                                            <div class="btn-group btn-group-sm">
                                                <button type="button" class="btn btn-info" data-bs-toggle="modal" 
                                                    data-bs-target="#orderDetailsModal" 
                                                    data-order-id="<?php echo $order['id']; ?>" 
                                                    onclick="showOrderDetails(<?php echo $order['id']; ?>)">
                                                    <i class="fas fa-eye"></i>
                                                </button>
                                                
                                                <?php if ($order['status'] === 'pending'): ?>
                                                    <button type="button" class="btn btn-success" data-bs-toggle="modal" 
                                                        data-bs-target="#changeStatusModal" 
                                                        data-order-id="<?php echo $order['id']; ?>" 
                                                        data-status="approved"
                                                        onclick="prepareStatusChange(<?php echo $order['id']; ?>, 'approved', 'تایید')">
                                                        <i class="fas fa-check"></i>
                                                    </button>
                                                    
                                                    <button type="button" class="btn btn-danger" data-bs-toggle="modal" 
                                                        data-bs-target="#changeStatusModal" 
                                                        data-order-id="<?php echo $order['id']; ?>" 
                                                        data-status="rejected"
                                                        onclick="prepareStatusChange(<?php echo $order['id']; ?>, 'rejected', 'رد')">
                                                        <i class="fas fa-times"></i>
                                                    </button>
                                                <?php endif; ?>
                                                
                                                <?php if ($order['status'] === 'approved'): ?>
                                                    <button type="button" class="btn btn-warning" data-bs-toggle="modal" 
                                                        data-bs-target="#changeStatusModal" 
                                                        data-order-id="<?php echo $order['id']; ?>" 
                                                        data-status="refunded"
                                                        onclick="prepareStatusChange(<?php echo $order['id']; ?>, 'refunded', 'استرداد')">
                                                        <i class="fas fa-undo"></i>
                                                    </button>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
                
                <!-- گزینه‌های انتخاب شده -->
                <div class="row mt-3">
                    <div class="col">
                        <div class="btn-group">
                            <button type="button" class="btn btn-outline-danger dropdown-toggle" data-bs-toggle="dropdown">
                                عملیات گروهی
                            </button>
                            <div class="dropdown-menu">
                                <button type="submit" class="dropdown-item text-danger" name="action" value="delete" 
                                    onclick="return confirm('آیا از حذف سفارش‌های انتخاب شده اطمینان دارید؟ این عمل غیرقابل بازگشت است!')">
                                    حذف سفارش‌ها
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </form>
            
            <!-- صفحه‌بندی -->
            <?php if ($total_pages > 1): ?>
                <nav aria-label="Page navigation" class="mt-4">
                    <ul class="pagination justify-content-center">
                        <?php if ($page > 1): ?>
                            <li class="page-item">
                                <a class="page-link" href="?page=orders&page_num=<?php echo $page - 1; ?>&search=<?php echo urlencode($search); ?>&status=<?php echo $filter_status; ?>">
                                    قبلی
                                </a>
                            </li>
                        <?php endif; ?>
                        
                        <?php for ($i = max(1, $page - 2); $i <= min($total_pages, $page + 2); $i++): ?>
                            <li class="page-item <?php echo $i === $page ? 'active' : ''; ?>">
                                <a class="page-link" href="?page=orders&page_num=<?php echo $i; ?>&search=<?php echo urlencode($search); ?>&status=<?php echo $filter_status; ?>">
                                    <?php echo $i; ?>
                                </a>
                            </li>
                        <?php endfor; ?>
                        
                        <?php if ($page < $total_pages): ?>
                            <li class="page-item">
                                <a class="page-link" href="?page=orders&page_num=<?php echo $page + 1; ?>&search=<?php echo urlencode($search); ?>&status=<?php echo $filter_status; ?>">
                                    بعدی
                                </a>
                            </li>
                        <?php endif; ?>
                    </ul>
                </nav>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- مودال تغییر وضعیت سفارش -->
<div class="modal fade" id="changeStatusModal" tabindex="-1" aria-labelledby="changeStatusModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <form action="" method="post">
                <input type="hidden" name="action" value="change_status">
                <input type="hidden" name="order_id" id="statusOrderId">
                <input type="hidden" name="status" id="newStatus">
                
                <div class="modal-header">
                    <h5 class="modal-title" id="statusModalTitle">تغییر وضعیت سفارش</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="admin_note" class="form-label">توضیحات (اختیاری)</label>
                        <textarea class="form-control" id="admin_note" name="admin_note" rows="3" 
                            placeholder="توضیحات خود را وارد کنید..."></textarea>
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">انصراف</button>
                    <button type="submit" class="btn btn-primary" id="confirmStatusBtn">تایید</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- مودال جزئیات سفارش -->
<div class="modal fade" id="orderDetailsModal" tabindex="-1" aria-labelledby="orderDetailsModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="orderDetailsModalLabel">جزئیات سفارش</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            
            <div class="modal-body" id="orderDetailsContent">
                <div class="text-center">
                    <div class="spinner-border" role="status">
                        <span class="visually-hidden">در حال بارگذاری...</span>
                    </div>
                </div>
            </div>
            
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">بستن</button>
            </div>
        </div>
    </div>
</div>

<script>
    // آماده‌سازی مودال تغییر وضعیت
    function prepareStatusChange(orderId, status, statusText) {
        document.getElementById('statusOrderId').value = orderId;
        document.getElementById('newStatus').value = status;
        document.getElementById('statusModalTitle').textContent = statusText + ' سفارش #' + orderId;
        
        const confirmBtn = document.getElementById('confirmStatusBtn');
        confirmBtn.textContent = statusText;
        
        // تنظیم کلاس دکمه بر اساس وضعیت
        confirmBtn.className = 'btn';
        switch (status) {
            case 'approved':
                confirmBtn.classList.add('btn-success');
                break;
            case 'rejected':
                confirmBtn.classList.add('btn-danger');
                break;
            case 'refunded':
                confirmBtn.classList.add('btn-warning');
                break;
            default:
                confirmBtn.classList.add('btn-primary');
        }
    }
    
    // نمایش جزئیات سفارش
    function showOrderDetails(orderId) {
        const detailsContent = document.getElementById('orderDetailsContent');
        detailsContent.innerHTML = '<div class="text-center"><div class="spinner-border" role="status"><span class="visually-hidden">در حال بارگذاری...</span></div></div>';
        
        // در اینجا می‌توانید از Ajax برای دریافت اطلاعات سفارش از سرور استفاده کنید
        // برای مثال:
        fetch(`admin/api/get_order_details.php?id=${orderId}`)
            .then(response => response.json())
            .then(data => {
                // نمایش اطلاعات سفارش در مودال
                let html = '';
                if (data.success) {
                    const order = data.order;
                    
                    html = `
                        <div class="row">
                            <div class="col-md-6">
                                <h6>اطلاعات اصلی</h6>
                                <table class="table table-striped">
                                    <tr>
                                        <th>شناسه سفارش</th>
                                        <td>#${order.id}</td>
                                    </tr>
                                    <tr>
                                        <th>کاربر</th>
                                        <td>${order.username}</td>
                                    </tr>
                                    <tr>
                                        <th>نوع سفارش</th>
                                        <td>${order.order_type_text}</td>
                                    </tr>
                                    <tr>
                                        <th>مبلغ</th>
                                        <td>${order.amount} تومان</td>
                                    </tr>
                                    <tr>
                                        <th>تاریخ ثبت</th>
                                        <td>${order.created_at}</td>
                                    </tr>
                                    <tr>
                                        <th>وضعیت</th>
                                        <td><span class="badge bg-${order.status_class}">${order.status_text}</span></td>
                                    </tr>
                                </table>
                            </div>
                            <div class="col-md-6">
                                <h6>جزئیات سرویس</h6>
                                <table class="table table-striped">
                                    <tr>
                                        <th>مدت زمان</th>
                                        <td>${order.duration} روز</td>
                                    </tr>
                                    <tr>
                                        <th>حجم دیتا</th>
                                        <td>${order.data_quota_formatted}</td>
                                    </tr>
                                    <tr>
                                        <th>تعداد دستگاه‌ها</th>
                                        <td>${order.max_devices}</td>
                                    </tr>
                                </table>
                                
                                <h6 class="mt-3">یادداشت‌ها</h6>
                                <div class="card">
                                    <div class="card-body">
                                        <p class="mb-0">${order.admin_note || 'بدون یادداشت'}</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="row mt-4">
                            <div class="col-12">
                                <h6>رسید پرداخت</h6>
                                ${order.payment_receipt ? `<div class="text-center"><img src="${order.payment_receipt}" class="img-fluid" style="max-height: 300px;"></div>` : '<div class="alert alert-info">رسید پرداختی آپلود نشده است.</div>'}
                            </div>
                        </div>
                    `;
                } else {
                    html = '<div class="alert alert-danger">خطا در دریافت اطلاعات سفارش.</div>';
                }
                
                detailsContent.innerHTML = html;
            })
            .catch(error => {
                detailsContent.innerHTML = '<div class="alert alert-danger">خطا در ارتباط با سرور.</div>';
            });
    }
    
    // انتخاب همه آیتم‌ها
    document.getElementById('selectAll').addEventListener('change', function() {
        const checkboxes = document.querySelectorAll('.item-checkbox');
        checkboxes.forEach(checkbox => {
            checkbox.checked = this.checked;
        });
    });
</script> 