<?php
require_once 'db.php';
require_once 'settings.php';

/**
 * Create a notification
 * 
 * @param string $title Notification title
 * @param string $message Notification message
 * @param string $type Notification type (info, warning, error, success)
 * @param int $user_id User ID (null for system notification)
 * @param string $sent_via Where to send notification (app, telegram, fcm, all)
 * @return int|false Notification ID if successful, false otherwise
 */
function create_notification($title, $message, $type = 'info', $user_id = null, $sent_via = 'all') {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            INSERT INTO notifications (user_id, title, message, type, sent_via) 
            VALUES (?, ?, ?, ?, ?)
        ");
        $stmt->execute([$user_id, $title, $message, $type, $sent_via]);
        
        $notification_id = $pdo->lastInsertId();
        
        // Send notification based on sent_via
        if ($sent_via === 'all' || $sent_via === 'telegram') {
            send_telegram_notification($user_id, $title, $message, $type);
        }
        
        if ($sent_via === 'all' || $sent_via === 'fcm') {
            send_fcm_notification($user_id, $title, $message, $type);
        }
        
        // Mark as sent
        $stmt = $pdo->prepare("UPDATE notifications SET is_sent = 1 WHERE id = ?");
        $stmt->execute([$notification_id]);
        
        return $notification_id;
    } catch (PDOException $e) {
        error_log("Error creating notification: " . $e->getMessage());
        return false;
    }
}

/**
 * Send notification via Telegram
 * 
 * @param int $user_id User ID (null for admin notification)
 * @param string $title Notification title
 * @param string $message Notification message
 * @param string $type Notification type
 * @return bool True if successful, false otherwise
 */
function send_telegram_notification($user_id, $title, $message, $type) {
    // Get Telegram bot token
    $bot_token = get_setting('telegram_bot_token');
    
    if (empty($bot_token)) {
        error_log("Telegram bot token not set");
        return false;
    }
    
    // Get chat ID
    $chat_id = null;
    
    if ($user_id === null) {
        // Send to admin
        $chat_id = get_setting('telegram_admin_id');
    } else {
        // Get user's Telegram ID
        global $pdo;
        $stmt = $pdo->prepare("SELECT telegram_id FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($user && !empty($user['telegram_id'])) {
            $chat_id = $user['telegram_id'];
        }
    }
    
    if (empty($chat_id)) {
        error_log("Telegram chat ID not found");
        return false;
    }
    
    // Format message
    $formatted_message = "*$title*\n\n$message";
    
    // Send message
    $url = "https://api.telegram.org/bot$bot_token/sendMessage";
    $data = [
        'chat_id' => $chat_id,
        'text' => $formatted_message,
        'parse_mode' => 'Markdown'
    ];
    
    $options = [
        'http' => [
            'header' => "Content-type: application/x-www-form-urlencoded\r\n",
            'method' => 'POST',
            'content' => http_build_query($data)
        ]
    ];
    
    $context = stream_context_create($options);
    $result = file_get_contents($url, false, $context);
    
    if ($result === false) {
        error_log("Error sending Telegram notification");
        return false;
    }
    
    return true;
}

/**
 * Send notification via FCM
 * 
 * @param int $user_id User ID
 * @param string $title Notification title
 * @param string $message Notification message
 * @param string $type Notification type
 * @return bool True if successful, false otherwise
 */
function send_fcm_notification($user_id, $title, $message, $type) {
    if ($user_id === null) {
        return false; // FCM requires a user ID
    }
    
    // Get FCM server key
    $server_key = get_setting('fcm_server_key');
    
    if (empty($server_key)) {
        error_log("FCM server key not set");
        return false;
    }
    
    // Get user's FCM tokens
    global $pdo;
    $stmt = $pdo->prepare("SELECT fcm_token FROM fcm_tokens WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $tokens = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    if (empty($tokens)) {
        error_log("No FCM tokens found for user $user_id");
        return false;
    }
    
    // Prepare notification data
    $notification = [
        'title' => $title,
        'body' => $message,
        'icon' => '/assets/images/notification-icon.png',
        'click_action' => 'OPEN_APP'
    ];
    
    $data = [
        'type' => $type,
        'message' => $message,
        'timestamp' => time()
    ];
    
    // Send to each token
    $success = false;
    
    foreach ($tokens as $token) {
        $fields = [
            'to' => $token,
            'notification' => $notification,
            'data' => $data
        ];
        
        $headers = [
            'Authorization: key=' . $server_key,
            'Content-Type: application/json'
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, 'https://fcm.googleapis.com/fcm/send');
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($fields));
        
        $result = curl_exec($ch);
        
        if ($result === false) {
            error_log("FCM error: " . curl_error($ch));
        } else {
            $success = true;
        }
        
        curl_close($ch);
    }
    
    return $success;
}

/**
 * Register FCM token for a user
 * 
 * @param int $user_id User ID
 * @param string $device_id Device ID
 * @param string $fcm_token FCM token
 * @return bool True if successful, false otherwise
 */
function register_fcm_token($user_id, $device_id, $fcm_token) {
    global $pdo;
    
    try {
        // Check if token already exists
        $stmt = $pdo->prepare("SELECT id FROM fcm_tokens WHERE user_id = ? AND device_id = ?");
        $stmt->execute([$user_id, $device_id]);
        $token = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($token) {
            // Update token
            $stmt = $pdo->prepare("UPDATE fcm_tokens SET fcm_token = ? WHERE id = ?");
            $stmt->execute([$fcm_token, $token['id']]);
        } else {
            // Insert new token
            $stmt = $pdo->prepare("INSERT INTO fcm_tokens (user_id, device_id, fcm_token) VALUES (?, ?, ?)");
            $stmt->execute([$user_id, $device_id, $fcm_token]);
        }
        
        return true;
    } catch (PDOException $e) {
        error_log("Error registering FCM token: " . $e->getMessage());
        return false;
    }
}

/**
 * Get user notifications
 * 
 * @param int $user_id User ID
 * @param int $limit Limit (optional)
 * @param bool $unread_only Get only unread notifications (optional)
 * @return array Array of notifications
 */
function get_user_notifications($user_id, $limit = 20, $unread_only = false) {
    global $pdo;
    
    try {
        $sql = "
            SELECT * FROM notifications 
            WHERE user_id = ?
        ";
        
        if ($unread_only) {
            $sql .= " AND is_read = 0";
        }
        
        $sql .= " ORDER BY created_at DESC LIMIT ?";
        
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(1, $user_id, PDO::PARAM_INT);
        $stmt->bindValue(2, $limit, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error getting user notifications: " . $e->getMessage());
        return [];
    }
}

/**
 * Mark notification as read
 * 
 * @param int $notification_id Notification ID
 * @return bool True if successful, false otherwise
 */
function mark_notification_read($notification_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("UPDATE notifications SET is_read = 1 WHERE id = ?");
        $stmt->execute([$notification_id]);
        
        return $stmt->rowCount() > 0;
    } catch (PDOException $e) {
        error_log("Error marking notification as read: " . $e->getMessage());
        return false;
    }
}

/**
 * Mark all user notifications as read
 * 
 * @param int $user_id User ID
 * @return bool True if successful, false otherwise
 */
function mark_all_notifications_read($user_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("UPDATE notifications SET is_read = 1 WHERE user_id = ?");
        $stmt->execute([$user_id]);
        
        return true;
    } catch (PDOException $e) {
        error_log("Error marking all notifications as read: " . $e->getMessage());
        return false;
    }
}

/**
 * Count unread notifications for a user
 * 
 * @param int $user_id User ID
 * @return int Number of unread notifications
 */
function count_unread_notifications($user_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM notifications WHERE user_id = ? AND is_read = 0");
        $stmt->execute([$user_id]);
        
        return (int) $stmt->fetchColumn();
    } catch (PDOException $e) {
        error_log("Error counting unread notifications: " . $e->getMessage());
        return 0;
    }
}

/**
 * Send low quota notification if needed
 * 
 * @param int $user_id User ID
 * @return bool True if notification sent, false otherwise
 */
function check_and_notify_low_quota($user_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT usage, quota FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$user || $user['quota'] == 0) {
            return false; // User not found or unlimited quota
        }
        
        $usage_percentage = ($user['usage'] / $user['quota']) * 100;
        
        // Check if usage is over 80% but less than 90%
        if ($usage_percentage >= 80 && $usage_percentage < 90) {
            // Check if we already sent a notification recently
            $stmt = $pdo->prepare("
                SELECT id FROM notifications 
                WHERE user_id = ? AND title LIKE '%حجم مصرفی%' AND created_at > DATE_SUB(NOW(), INTERVAL 1 DAY)
            ");
            $stmt->execute([$user_id]);
            
            if (!$stmt->fetch()) {
                // Send notification
                $remaining_gb = round(($user['quota'] - $user['usage']) / 1073741824, 2);
                create_notification(
                    'هشدار حجم مصرفی',
                    "حجم مصرفی شما به بیش از 80% رسیده است. حجم باقیمانده: {$remaining_gb} گیگابایت",
                    'warning',
                    $user_id
                );
                return true;
            }
        }
        // Check if usage is over 90%
        else if ($usage_percentage >= 90) {
            // Check if we already sent a notification recently
            $stmt = $pdo->prepare("
                SELECT id FROM notifications 
                WHERE user_id = ? AND title LIKE '%حجم مصرفی%' AND created_at > DATE_SUB(NOW(), INTERVAL 12 HOUR)
            ");
            $stmt->execute([$user_id]);
            
            if (!$stmt->fetch()) {
                // Send notification
                $remaining_gb = round(($user['quota'] - $user['usage']) / 1073741824, 2);
                create_notification(
                    'هشدار حجم مصرفی',
                    "حجم مصرفی شما به بیش از 90% رسیده است. حجم باقیمانده: {$remaining_gb} گیگابایت",
                    'error',
                    $user_id
                );
                return true;
            }
        }
        
        return false;
    } catch (PDOException $e) {
        error_log("Error checking quota for notification: " . $e->getMessage());
        return false;
    }
}

/**
 * Send expiration notification if needed
 * 
 * @param int $user_id User ID
 * @return bool True if notification sent, false otherwise
 */
function check_and_notify_expiration($user_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT expiration FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$user || $user['expiration'] === null) {
            return false; // User not found or no expiration
        }
        
        $expiration_time = strtotime($user['expiration']);
        $days_remaining = ceil(($expiration_time - time()) / 86400);
        
        // Check if expiration is in 3 days or less
        if ($days_remaining <= 3 && $days_remaining > 0) {
            // Check if we already sent a notification recently
            $stmt = $pdo->prepare("
                SELECT id FROM notifications 
                WHERE user_id = ? AND title LIKE '%انقضای اشتراک%' AND created_at > DATE_SUB(NOW(), INTERVAL 1 DAY)
            ");
            $stmt->execute([$user_id]);
            
            if (!$stmt->fetch()) {
                // Send notification
                create_notification(
                    'هشدار انقضای اشتراک',
                    "اشتراک شما {$days_remaining} روز دیگر منقضی می‌شود. لطفاً نسبت به تمدید آن اقدام نمایید.",
                    'warning',
                    $user_id
                );
                return true;
            }
        }
        
        return false;
    } catch (PDOException $e) {
        error_log("Error checking expiration for notification: " . $e->getMessage());
        return false;
    }
}
?> 