<?php
require_once 'db.php';
require_once 'settings.php';
require_once 'user.php';
require_once 'notification.php';

/**
 * Create a new order
 * 
 * @param int $user_id User ID
 * @param string $order_type Order type (new, renewal, upgrade)
 * @param float $amount Amount
 * @param int $duration Duration in days
 * @param int $data_quota Data quota in bytes
 * @param int $max_devices Maximum number of devices
 * @param string $payment_receipt Payment receipt (optional)
 * @return int|false Order ID if successful, false otherwise
 */
function create_order($user_id, $order_type, $amount, $duration, $data_quota, $max_devices, $payment_receipt = null) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            INSERT INTO orders (user_id, order_type, amount, duration, data_quota, max_devices, payment_receipt) 
            VALUES (?, ?, ?, ?, ?, ?, ?)
        ");
        
        $stmt->execute([
            $user_id, 
            $order_type, 
            $amount, 
            $duration, 
            $data_quota, 
            $max_devices, 
            $payment_receipt
        ]);
        
        $order_id = $pdo->lastInsertId();
        
        // Notify admin about new order
        create_notification(
            'سفارش جدید',
            "یک سفارش جدید با شناسه $order_id ثبت شده است.",
            'info',
            null, // Send to admin
            'telegram' // Only via Telegram
        );
        
        return $order_id;
    } catch (PDOException $e) {
        error_log("Error creating order: " . $e->getMessage());
        return false;
    }
}

/**
 * Upload payment receipt
 * 
 * @param int $order_id Order ID
 * @param array $file File data from $_FILES
 * @return bool True if successful, false otherwise
 */
function upload_payment_receipt($order_id, $file) {
    // Check if file is valid
    if (!isset($file['tmp_name']) || !is_uploaded_file($file['tmp_name'])) {
        return false;
    }
    
    // Create directory if not exists
    $upload_dir = '../uploads/receipts/';
    if (!file_exists($upload_dir)) {
        mkdir($upload_dir, 0755, true);
    }
    
    // Generate unique filename
    $filename = $order_id . '_' . time() . '_' . basename($file['name']);
    $filepath = $upload_dir . $filename;
    
    // Move uploaded file
    if (move_uploaded_file($file['tmp_name'], $filepath)) {
        // Update order with receipt path
        global $pdo;
        $stmt = $pdo->prepare("UPDATE orders SET payment_receipt = ? WHERE id = ?");
        $stmt->execute([$filename, $order_id]);
        
        return $stmt->rowCount() > 0;
    }
    
    return false;
}

/**
 * Get order by ID
 * 
 * @param int $order_id Order ID
 * @return array|false Order data or false if not found
 */
function get_order($order_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT * FROM orders WHERE id = ?");
        $stmt->execute([$order_id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error getting order: " . $e->getMessage());
        return false;
    }
}

/**
 * Get user orders
 * 
 * @param int $user_id User ID
 * @param int $limit Limit (optional)
 * @return array Array of orders
 */
function get_user_orders($user_id, $limit = 10) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT * FROM orders 
            WHERE user_id = ? 
            ORDER BY created_at DESC 
            LIMIT ?
        ");
        $stmt->bindValue(1, $user_id, PDO::PARAM_INT);
        $stmt->bindValue(2, $limit, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error getting user orders: " . $e->getMessage());
        return [];
    }
}

/**
 * Get all pending orders
 * 
 * @param int $limit Limit (optional)
 * @return array Array of pending orders
 */
function get_pending_orders($limit = 20) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT o.*, u.username 
            FROM orders o 
            JOIN users u ON o.user_id = u.id 
            WHERE o.status = 'pending' 
            ORDER BY o.created_at ASC 
            LIMIT ?
        ");
        $stmt->bindValue(1, $limit, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error getting pending orders: " . $e->getMessage());
        return [];
    }
}

/**
 * Update order status
 * 
 * @param int $order_id Order ID
 * @param string $status New status (pending, paid, rejected, completed)
 * @param string $admin_note Admin note (optional)
 * @return bool True if successful, false otherwise
 */
function update_order_status($order_id, $status, $admin_note = null) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            UPDATE orders 
            SET status = ?, admin_note = ? 
            WHERE id = ?
        ");
        $stmt->execute([$status, $admin_note, $order_id]);
        
        if ($stmt->rowCount() > 0) {
            // Get order details
            $order = get_order($order_id);
            
            if ($order) {
                // Notify user about order status change
                $message = '';
                $type = 'info';
                
                switch ($status) {
                    case 'paid':
                        $message = "پرداخت سفارش شما با شناسه {$order_id} تایید شد و در حال پردازش است.";
                        $type = 'success';
                        break;
                    case 'rejected':
                        $message = "متاسفانه سفارش شما با شناسه {$order_id} رد شد. دلیل: " . ($admin_note ?: 'نامشخص');
                        $type = 'error';
                        break;
                    case 'completed':
                        $message = "سفارش شما با شناسه {$order_id} با موفقیت تکمیل شد.";
                        $type = 'success';
                        
                        // Apply order to user account
                        apply_order_to_user($order_id);
                        break;
                }
                
                if (!empty($message)) {
                    create_notification(
                        'بروزرسانی وضعیت سفارش',
                        $message,
                        $type,
                        $order['user_id']
                    );
                }
            }
            
            return true;
        }
        
        return false;
    } catch (PDOException $e) {
        error_log("Error updating order status: " . $e->getMessage());
        return false;
    }
}

/**
 * Apply order to user account
 * 
 * @param int $order_id Order ID
 * @return bool True if successful, false otherwise
 */
function apply_order_to_user($order_id) {
    global $pdo;
    
    try {
        // Get order details
        $order = get_order($order_id);
        
        if (!$order || $order['status'] !== 'completed') {
            return false;
        }
        
        // Get user details
        $user = get_user_by_id($order['user_id']);
        
        if (!$user) {
            return false;
        }
        
        // Update user based on order type
        $update_data = [];
        
        switch ($order['order_type']) {
            case 'new':
                // For new accounts, set expiration and quota directly
                $expiration = date('Y-m-d H:i:s', strtotime("+{$order['duration']} days"));
                $update_data = [
                    'expiration' => $expiration,
                    'quota' => $order['data_quota'],
                    'max_devices' => $order['max_devices'],
                    'is_blocked' => 0
                ];
                break;
                
            case 'renewal':
                // For renewals, extend expiration and add quota
                $current_expiration = $user['expiration'] ? $user['expiration'] : date('Y-m-d H:i:s');
                $new_expiration = date('Y-m-d H:i:s', strtotime($current_expiration . " +{$order['duration']} days"));
                $update_data = [
                    'expiration' => $new_expiration,
                    'quota' => $user['quota'] + $order['data_quota'],
                    'is_blocked' => 0
                ];
                break;
                
            case 'upgrade':
                // For upgrades, update max_devices and possibly add quota
                $update_data = [
                    'max_devices' => $order['max_devices']
                ];
                
                if ($order['data_quota'] > 0) {
                    $update_data['quota'] = $user['quota'] + $order['data_quota'];
                }
                break;
        }
        
        // Update user
        if (!empty($update_data)) {
            return update_user($order['user_id'], $update_data);
        }
        
        return false;
    } catch (PDOException $e) {
        error_log("Error applying order to user: " . $e->getMessage());
        return false;
    }
}

/**
 * Count pending orders
 * 
 * @return int Number of pending orders
 */
function count_pending_orders() {
    global $pdo;
    
    try {
        $stmt = $pdo->query("SELECT COUNT(*) FROM orders WHERE status = 'pending'");
        return (int) $stmt->fetchColumn();
    } catch (PDOException $e) {
        error_log("Error counting pending orders: " . $e->getMessage());
        return 0;
    }
}

/**
 * Get order statistics
 * 
 * @param string $period Period (today, week, month, all)
 * @return array Order statistics
 */
function get_order_statistics($period = 'all') {
    global $pdo;
    
    try {
        $where_clause = '';
        
        switch ($period) {
            case 'today':
                $where_clause = "WHERE DATE(created_at) = CURDATE()";
                break;
            case 'week':
                $where_clause = "WHERE created_at >= DATE_SUB(NOW(), INTERVAL 1 WEEK)";
                break;
            case 'month':
                $where_clause = "WHERE created_at >= DATE_SUB(NOW(), INTERVAL 1 MONTH)";
                break;
        }
        
        $sql = "
            SELECT 
                COUNT(*) as total_orders,
                SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending_orders,
                SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed_orders,
                SUM(CASE WHEN status = 'rejected' THEN 1 ELSE 0 END) as rejected_orders,
                SUM(amount) as total_amount
            FROM orders
            $where_clause
        ";
        
        $stmt = $pdo->query($sql);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error getting order statistics: " . $e->getMessage());
        return [
            'total_orders' => 0,
            'pending_orders' => 0,
            'completed_orders' => 0,
            'rejected_orders' => 0,
            'total_amount' => 0
        ];
    }
}
?> 