<?php
require_once 'db.php';

/**
 * Get a setting value from the database
 * 
 * @param string $key The setting key
 * @param mixed $default Default value if setting not found
 * @return mixed The setting value or default if not found
 */
function get_setting($key, $default = null) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT setting_value, setting_type FROM settings WHERE setting_key = ?");
        $stmt->execute([$key]);
        $setting = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$setting) {
            return $default;
        }
        
        // Convert value based on type
        switch ($setting['setting_type']) {
            case 'number':
                return (float) $setting['setting_value'];
            case 'boolean':
                return (bool) $setting['setting_value'];
            case 'json':
                return json_decode($setting['setting_value'], true);
            default:
                return $setting['setting_value'];
        }
    } catch (PDOException $e) {
        error_log("Error getting setting $key: " . $e->getMessage());
        return $default;
    }
}

/**
 * Update a setting value in the database
 * 
 * @param string $key The setting key
 * @param mixed $value The new value
 * @return bool True if successful, false otherwise
 */
function update_setting($key, $value) {
    global $pdo;
    
    try {
        // Get current setting type
        $stmt = $pdo->prepare("SELECT setting_type FROM settings WHERE setting_key = ?");
        $stmt->execute([$key]);
        $setting = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$setting) {
            return false;
        }
        
        // Convert value based on type before saving
        if ($setting['setting_type'] === 'json' && !is_string($value)) {
            $value = json_encode($value);
        }
        
        $stmt = $pdo->prepare("UPDATE settings SET setting_value = ? WHERE setting_key = ?");
        $stmt->execute([$value, $key]);
        
        return $stmt->rowCount() > 0;
    } catch (PDOException $e) {
        error_log("Error updating setting $key: " . $e->getMessage());
        return false;
    }
}

/**
 * Get all settings by category
 * 
 * @param string $category The settings category
 * @return array Array of settings
 */
function get_settings_by_category($category) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT setting_key, setting_value, setting_type, title, description FROM settings WHERE category = ?");
        $stmt->execute([$category]);
        $settings = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $result = [];
        foreach ($settings as $setting) {
            // Convert value based on type
            switch ($setting['setting_type']) {
                case 'number':
                    $value = (float) $setting['setting_value'];
                    break;
                case 'boolean':
                    $value = (bool) $setting['setting_value'];
                    break;
                case 'json':
                    $value = json_decode($setting['setting_value'], true);
                    break;
                default:
                    $value = $setting['setting_value'];
            }
            
            $result[$setting['setting_key']] = [
                'value' => $value,
                'type' => $setting['setting_type'],
                'title' => $setting['title'],
                'description' => $setting['description']
            ];
        }
        
        return $result;
    } catch (PDOException $e) {
        error_log("Error getting settings for category $category: " . $e->getMessage());
        return [];
    }
}

/**
 * Get all app settings as an array
 * 
 * @return array Array of all settings
 */
function get_all_settings() {
    global $pdo;
    
    try {
        $stmt = $pdo->query("SELECT setting_key, setting_value, setting_type, category FROM settings");
        $settings = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $result = [];
        foreach ($settings as $setting) {
            // Convert value based on type
            switch ($setting['setting_type']) {
                case 'number':
                    $value = (float) $setting['setting_value'];
                    break;
                case 'boolean':
                    $value = (bool) $setting['setting_value'];
                    break;
                case 'json':
                    $value = json_decode($setting['setting_value'], true);
                    break;
                default:
                    $value = $setting['setting_value'];
            }
            
            if (!isset($result[$setting['category']])) {
                $result[$setting['category']] = [];
            }
            
            $result[$setting['category']][$setting['setting_key']] = $value;
        }
        
        return $result;
    } catch (PDOException $e) {
        error_log("Error getting all settings: " . $e->getMessage());
        return [];
    }
}
?> 