<?php
require_once 'db.php';
require_once 'settings.php';

/**
 * Get user by ID
 * 
 * @param int $id User ID
 * @return array|false User data or false if not found
 */
function get_user_by_id($id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
        $stmt->execute([$id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error getting user by ID: " . $e->getMessage());
        return false;
    }
}

/**
 * Get user by username
 * 
 * @param string $username Username
 * @return array|false User data or false if not found
 */
function get_user_by_username($username) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT * FROM users WHERE username = ?");
        $stmt->execute([$username]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error getting user by username: " . $e->getMessage());
        return false;
    }
}

/**
 * Create a new user
 * 
 * @param string $username Username
 * @param string $password Plain text password
 * @param string $email Email (optional)
 * @param string $telegram_id Telegram ID (optional)
 * @param bool $is_test_account Whether this is a test account
 * @return int|false User ID if successful, false otherwise
 */
function create_user($username, $password, $email = null, $telegram_id = null, $is_test_account = false) {
    global $pdo;
    
    try {
        // Check if username already exists
        $stmt = $pdo->prepare("SELECT id FROM users WHERE username = ?");
        $stmt->execute([$username]);
        if ($stmt->fetch()) {
            return false; // Username already exists
        }
        
        // Set expiration and quota based on whether it's a test account
        $expiration = null;
        $quota = 0;
        $max_devices = 1;
        
        if ($is_test_account) {
            $test_duration = get_setting('test_account_duration', 1);
            $expiration = date('Y-m-d H:i:s', strtotime("+{$test_duration} days"));
            $quota = get_setting('test_account_quota', 1073741824); // Default 1GB
            $max_devices = get_setting('test_account_max_devices', 1);
        }
        
        // Hash password
        $hashed_password = password_hash($password, PASSWORD_DEFAULT);
        
        // Insert user
        $stmt = $pdo->prepare("
            INSERT INTO users (username, password, email, telegram_id, expiration, quota, max_devices, is_test_account) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        $stmt->execute([
            $username, 
            $hashed_password, 
            $email, 
            $telegram_id, 
            $expiration, 
            $quota, 
            $max_devices, 
            $is_test_account ? 1 : 0
        ]);
        
        return $pdo->lastInsertId();
    } catch (PDOException $e) {
        error_log("Error creating user: " . $e->getMessage());
        return false;
    }
}

/**
 * Update user data
 * 
 * @param int $user_id User ID
 * @param array $data Data to update
 * @return bool True if successful, false otherwise
 */
function update_user($user_id, $data) {
    global $pdo;
    
    try {
        $allowed_fields = [
            'email', 'telegram_id', 'expiration', 'quota', 
            'max_devices', 'is_blocked', 'is_test_account'
        ];
        
        $updates = [];
        $values = [];
        
        foreach ($data as $field => $value) {
            if (in_array($field, $allowed_fields)) {
                $updates[] = "$field = ?";
                $values[] = $value;
            }
        }
        
        if (empty($updates)) {
            return false; // No valid fields to update
        }
        
        // Add user_id to values
        $values[] = $user_id;
        
        $sql = "UPDATE users SET " . implode(', ', $updates) . " WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($values);
        
        return $stmt->rowCount() > 0;
    } catch (PDOException $e) {
        error_log("Error updating user: " . $e->getMessage());
        return false;
    }
}

/**
 * Change user password
 * 
 * @param int $user_id User ID
 * @param string $new_password New password
 * @return bool True if successful, false otherwise
 */
function change_password($user_id, $new_password) {
    global $pdo;
    
    try {
        $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
        
        $stmt = $pdo->prepare("UPDATE users SET password = ? WHERE id = ?");
        $stmt->execute([$hashed_password, $user_id]);
        
        return $stmt->rowCount() > 0;
    } catch (PDOException $e) {
        error_log("Error changing password: " . $e->getMessage());
        return false;
    }
}

/**
 * Update user usage
 * 
 * @param int $user_id User ID
 * @param int $bytes_used Bytes used
 * @return bool True if successful, false otherwise
 */
function update_user_usage($user_id, $bytes_used) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("UPDATE users SET usage = usage + ? WHERE id = ?");
        $stmt->execute([$bytes_used, $user_id]);
        
        return $stmt->rowCount() > 0;
    } catch (PDOException $e) {
        error_log("Error updating user usage: " . $e->getMessage());
        return false;
    }
}

/**
 * Check if user has exceeded quota
 * 
 * @param int $user_id User ID
 * @return bool True if quota exceeded, false otherwise
 */
function is_quota_exceeded($user_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT usage, quota FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$user) {
            return true; // User not found, consider quota exceeded
        }
        
        // If quota is 0, it means unlimited
        if ($user['quota'] == 0) {
            return false;
        }
        
        return $user['usage'] >= $user['quota'];
    } catch (PDOException $e) {
        error_log("Error checking quota: " . $e->getMessage());
        return true; // Consider quota exceeded on error
    }
}

/**
 * Check if user account has expired
 * 
 * @param int $user_id User ID
 * @return bool True if expired, false otherwise
 */
function is_account_expired($user_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT expiration FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$user) {
            return true; // User not found, consider expired
        }
        
        // If expiration is null, it means never expires
        if ($user['expiration'] === null) {
            return false;
        }
        
        return strtotime($user['expiration']) < time();
    } catch (PDOException $e) {
        error_log("Error checking expiration: " . $e->getMessage());
        return true; // Consider expired on error
    }
}

/**
 * Get all users with pagination
 * 
 * @param int $page Page number (1-based)
 * @param int $per_page Items per page
 * @return array Array of users
 */
function get_all_users($page = 1, $per_page = 20) {
    global $pdo;
    
    try {
        $offset = ($page - 1) * $per_page;
        
        $stmt = $pdo->prepare("
            SELECT * FROM users 
            ORDER BY id DESC 
            LIMIT ? OFFSET ?
        ");
        $stmt->bindValue(1, $per_page, PDO::PARAM_INT);
        $stmt->bindValue(2, $offset, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error getting users: " . $e->getMessage());
        return [];
    }
}

/**
 * Count total users
 * 
 * @return int Total number of users
 */
function count_users() {
    global $pdo;
    
    try {
        $stmt = $pdo->query("SELECT COUNT(*) FROM users");
        return (int) $stmt->fetchColumn();
    } catch (PDOException $e) {
        error_log("Error counting users: " . $e->getMessage());
        return 0;
    }
}

/**
 * Register device for a user
 * 
 * @param int $user_id User ID
 * @param string $device_id Device ID
 * @param string $device_name Device name (optional)
 * @param string $device_type Device type (optional)
 * @return bool True if successful, false otherwise
 */
function register_device($user_id, $device_id, $device_name = null, $device_type = null) {
    global $pdo;
    
    try {
        // Check if user has reached max devices
        $stmt = $pdo->prepare("
            SELECT u.max_devices, COUNT(d.id) as active_devices 
            FROM users u 
            LEFT JOIN devices d ON u.id = d.user_id AND d.is_active = 1 
            WHERE u.id = ? 
            GROUP BY u.id
        ");
        $stmt->execute([$user_id]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$result) {
            return false; // User not found
        }
        
        // Check if this device is already registered
        $stmt = $pdo->prepare("SELECT id FROM devices WHERE user_id = ? AND device_id = ?");
        $stmt->execute([$user_id, $device_id]);
        $existing_device = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($existing_device) {
            // Update last connection time
            $stmt = $pdo->prepare("
                UPDATE devices 
                SET last_connection = NOW(), device_name = ?, device_type = ?, is_active = 1 
                WHERE id = ?
            ");
            $stmt->execute([$device_name, $device_type, $existing_device['id']]);
            
            // Update active devices count
            $stmt = $pdo->prepare("
                UPDATE users 
                SET active_devices = (
                    SELECT COUNT(*) FROM devices 
                    WHERE user_id = ? AND is_active = 1
                ) 
                WHERE id = ?
            ");
            $stmt->execute([$user_id, $user_id]);
            
            return true;
        }
        
        // Check if max devices reached
        if ($result['active_devices'] >= $result['max_devices']) {
            return false; // Max devices reached
        }
        
        // Register new device
        $stmt = $pdo->prepare("
            INSERT INTO devices (user_id, device_id, device_name, device_type, last_connection) 
            VALUES (?, ?, ?, ?, NOW())
        ");
        $stmt->execute([$user_id, $device_id, $device_name, $device_type]);
        
        // Update active devices count
        $stmt = $pdo->prepare("
            UPDATE users 
            SET active_devices = (
                SELECT COUNT(*) FROM devices 
                WHERE user_id = ? AND is_active = 1
            ) 
            WHERE id = ?
        ");
        $stmt->execute([$user_id, $user_id]);
        
        return true;
    } catch (PDOException $e) {
        error_log("Error registering device: " . $e->getMessage());
        return false;
    }
}

/**
 * Deactivate a device
 * 
 * @param int $user_id User ID
 * @param string $device_id Device ID
 * @return bool True if successful, false otherwise
 */
function deactivate_device($user_id, $device_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            UPDATE devices 
            SET is_active = 0 
            WHERE user_id = ? AND device_id = ?
        ");
        $stmt->execute([$user_id, $device_id]);
        
        // Update active devices count
        $stmt = $pdo->prepare("
            UPDATE users 
            SET active_devices = (
                SELECT COUNT(*) FROM devices 
                WHERE user_id = ? AND is_active = 1
            ) 
            WHERE id = ?
        ");
        $stmt->execute([$user_id, $user_id]);
        
        return true;
    } catch (PDOException $e) {
        error_log("Error deactivating device: " . $e->getMessage());
        return false;
    }
}

/**
 * Get user devices
 * 
 * @param int $user_id User ID
 * @return array Array of devices
 */
function get_user_devices($user_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT * FROM devices 
            WHERE user_id = ? 
            ORDER BY last_connection DESC
        ");
        $stmt->execute([$user_id]);
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error getting user devices: " . $e->getMessage());
        return [];
    }
}

/**
 * Log connection
 * 
 * @param int $user_id User ID
 * @param string $device_id Device ID
 * @param string $ip_address IP address
 * @param string $location Location (optional)
 * @return int|false Connection log ID if successful, false otherwise
 */
function log_connection($user_id, $device_id, $ip_address, $location = null) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            INSERT INTO connection_logs (user_id, device_id, connection_time, ip_address, location) 
            VALUES (?, ?, NOW(), ?, ?)
        ");
        $stmt->execute([$user_id, $device_id, $ip_address, $location]);
        
        return $pdo->lastInsertId();
    } catch (PDOException $e) {
        error_log("Error logging connection: " . $e->getMessage());
        return false;
    }
}

/**
 * Log disconnection
 * 
 * @param int $log_id Connection log ID
 * @param int $data_used Data used in bytes
 * @return bool True if successful, false otherwise
 */
function log_disconnection($log_id, $data_used) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            UPDATE connection_logs 
            SET disconnection_time = NOW(), data_used = ? 
            WHERE id = ?
        ");
        $stmt->execute([$data_used, $log_id]);
        
        // Get user ID from log
        $stmt = $pdo->prepare("SELECT user_id FROM connection_logs WHERE id = ?");
        $stmt->execute([$log_id]);
        $log = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($log) {
            // Update user usage
            update_user_usage($log['user_id'], $data_used);
        }
        
        return true;
    } catch (PDOException $e) {
        error_log("Error logging disconnection: " . $e->getMessage());
        return false;
    }
}

/**
 * Get user connection logs
 * 
 * @param int $user_id User ID
 * @param int $limit Limit (optional)
 * @return array Array of connection logs
 */
function get_user_connection_logs($user_id, $limit = 50) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT * FROM connection_logs 
            WHERE user_id = ? 
            ORDER BY connection_time DESC 
            LIMIT ?
        ");
        $stmt->bindValue(1, $user_id, PDO::PARAM_INT);
        $stmt->bindValue(2, $limit, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Error getting user connection logs: " . $e->getMessage());
        return [];
    }
}
?> 