-- V2Ray Management System Database Schema

-- Settings table for dynamic configuration
CREATE TABLE settings (
    id INT AUTO_INCREMENT PRIMARY KEY,
    setting_key VARCHAR(50) NOT NULL UNIQUE,
    setting_value TEXT,
    setting_type ENUM('text', 'number', 'boolean', 'json', 'color', 'image') NOT NULL DEFAULT 'text',
    category ENUM('app', 'theme', 'test_account', 'telegram', 'payment', 'about') NOT NULL,
    title VARCHAR(100) NOT NULL,
    description VARCHAR(255),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Users table
CREATE TABLE users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) NOT NULL UNIQUE,
    password VARCHAR(255) NOT NULL,
    email VARCHAR(100),
    telegram_id VARCHAR(50),
    role ENUM('admin', 'user') NOT NULL DEFAULT 'user',
    expiration DATETIME,
    quota BIGINT DEFAULT 0, -- Total allowed data in bytes
    usage BIGINT DEFAULT 0, -- Used data in bytes
    active_devices INT DEFAULT 0,
    max_devices INT DEFAULT 1,
    is_blocked TINYINT(1) DEFAULT 0,
    is_test_account TINYINT(1) DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Devices table
CREATE TABLE devices (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    device_id VARCHAR(100) NOT NULL,
    device_name VARCHAR(100),
    device_type VARCHAR(50),
    last_connection DATETIME,
    is_active TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY (user_id, device_id)
);

-- Connection logs
CREATE TABLE connection_logs (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    device_id VARCHAR(100),
    connection_time DATETIME NOT NULL,
    disconnection_time DATETIME,
    data_used BIGINT DEFAULT 0, -- Data used in this session in bytes
    ip_address VARCHAR(45),
    location VARCHAR(100),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

-- Orders table
CREATE TABLE orders (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    order_type ENUM('new', 'renewal', 'upgrade') NOT NULL,
    amount DECIMAL(10, 2) NOT NULL,
    duration INT NOT NULL, -- Duration in days
    data_quota BIGINT NOT NULL, -- Data quota in bytes
    max_devices INT NOT NULL,
    status ENUM('pending', 'paid', 'rejected', 'completed') NOT NULL DEFAULT 'pending',
    payment_receipt TEXT, -- Path to uploaded receipt image or text description
    admin_note TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

-- Notifications table
CREATE TABLE notifications (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT,
    title VARCHAR(100) NOT NULL,
    message TEXT NOT NULL,
    type ENUM('info', 'warning', 'error', 'success') NOT NULL DEFAULT 'info',
    is_read TINYINT(1) DEFAULT 0,
    is_sent TINYINT(1) DEFAULT 0,
    sent_via ENUM('app', 'telegram', 'fcm', 'all') NOT NULL DEFAULT 'all',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

-- FCM tokens for push notifications
CREATE TABLE fcm_tokens (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    device_id VARCHAR(100) NOT NULL,
    fcm_token TEXT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY (user_id, device_id)
);

-- Insert default settings
INSERT INTO settings (setting_key, setting_value, setting_type, category, title, description) VALUES
('app_name', 'V2Ray Manager', 'text', 'app', 'نام برنامه', 'نام نمایشی برنامه'),
('app_description', 'سیستم مدیریت سرویس V2Ray', 'text', 'app', 'توضیحات برنامه', 'توضیحات کوتاه درباره برنامه'),
('app_logo', '/assets/images/logo.png', 'image', 'app', 'لوگوی برنامه', 'آدرس تصویر لوگوی برنامه'),
('primary_color', '#1976d2', 'color', 'theme', 'رنگ اصلی', 'رنگ اصلی تم برنامه'),
('secondary_color', '#f50057', 'color', 'theme', 'رنگ ثانویه', 'رنگ ثانویه تم برنامه'),
('background_color', '#f5f5f5', 'color', 'theme', 'رنگ پس زمینه', 'رنگ پس زمینه برنامه'),
('test_account_duration', '1', 'number', 'test_account', 'مدت اعتبار حساب تست', 'مدت اعتبار حساب تست به روز'),
('test_account_quota', '1073741824', 'number', 'test_account', 'حجم حساب تست', 'حجم حساب تست به بایت (1GB)'),
('test_account_max_devices', '1', 'number', 'test_account', 'حداکثر دستگاه‌های حساب تست', 'حداکثر تعداد دستگاه‌های مجاز برای حساب تست'),
('telegram_bot_token', '', 'text', 'telegram', 'توکن ربات تلگرام', 'توکن API ربات تلگرام'),
('telegram_admin_id', '', 'text', 'telegram', 'شناسه تلگرام مدیر', 'شناسه عددی تلگرام مدیر برای دریافت اعلان‌ها'),
('payment_info', 'لطفا مبلغ را به شماره کارت زیر واریز کنید:\n6037-XXXX-XXXX-XXXX\nبه نام: نام صاحب کارت', 'text', 'payment', 'اطلاعات پرداخت', 'اطلاعات پرداخت که به کاربر نمایش داده می‌شود'),
('about_developer', 'توسعه داده شده توسط: نام توسعه دهنده', 'text', 'about', 'درباره توسعه دهنده', 'اطلاعات درباره توسعه دهنده یا تیم'),
('about_project', 'این پروژه یک سیستم مدیریت V2Ray است که امکان مدیریت کاربران و سرویس‌ها را فراهم می‌کند.', 'text', 'about', 'درباره پروژه', 'توضیحات درباره پروژه'),
('fcm_server_key', '', 'text', 'app', 'کلید سرور FCM', 'کلید سرور Firebase Cloud Messaging برای ارسال اعلان‌ها');

-- Create admin user with password 'admin123'
INSERT INTO users (username, password, role, quota, max_devices) VALUES
('admin', '$2y$10$8MNXAOYLSbWYRGX1jHIUEuiCBmJ4KAj9bIQQCJBwP0RI8by6.Lyiq', 'admin', 1099511627776, 10); 