<?php
/**
 * ربات تلگرام برای سیستم مدیریت V2Ray
 * این فایل درخواست‌های ارسالی از تلگرام را پردازش می‌کند
 */

// تنظیم اینکه خطاها نمایش داده شود (در محیط تولید باید غیرفعال شود)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// فراخوانی فایل‌های مورد نیاز
require_once '../config/db.php';
require_once '../config/settings.php';
require_once '../config/user.php';
require_once '../config/order.php';
require_once '../config/notification.php';

// دریافت توکن ربات از تنظیمات
$bot_token = get_setting('telegram_bot_token');
$admin_id = get_setting('telegram_admin_id');

// بررسی اینکه توکن ربات تنظیم شده باشد
if (empty($bot_token)) {
    die("توکن ربات تلگرام تنظیم نشده است");
}

// دریافت اطلاعات ارسال شده توسط تلگرام
$update = json_decode(file_get_contents('php://input'), true);

// ذخیره لاگ درخواست‌ها (برای اشکال‌زدایی)
file_put_contents('../logs/telegram_update.log', date('Y-m-d H:i:s') . ': ' . json_encode($update) . PHP_EOL, FILE_APPEND);

// بررسی نوع پیام دریافتی
if (isset($update['message'])) {
    $message = $update['message'];
    $chat_id = $message['chat']['id'];
    $text = $message['text'] ?? '';
    $user = $message['from'];
    
    // پردازش دستورات
    if (isset($message['entities']) && $message['entities'][0]['type'] === 'bot_command') {
        processCommand($text, $chat_id, $user);
    } 
    // پردازش عکس (برای رسید پرداخت)
    elseif (isset($message['photo'])) {
        processPhoto($message, $chat_id, $user);
    }
    // پردازش متن عادی
    else {
        processText($text, $chat_id, $user);
    }
} 
// پردازش کالبک‌های دکمه‌ها
elseif (isset($update['callback_query'])) {
    $callback_query = $update['callback_query'];
    $chat_id = $callback_query['message']['chat']['id'];
    $callback_data = $callback_query['data'];
    $user = $callback_query['from'];
    
    processCallback($callback_data, $chat_id, $user, $callback_query['id']);
}

/**
 * پردازش دستورات ربات
 * 
 * @param string $command دستور
 * @param int $chat_id شناسه چت
 * @param array $user اطلاعات کاربر فرستنده
 */
function processCommand($command, $chat_id, $user) {
    switch ($command) {
        case '/start':
            sendStartMessage($chat_id);
            break;
        
        case '/status':
            sendStatusMessage($chat_id, $user);
            break;
            
        case '/test':
            createTestAccount($chat_id, $user);
            break;
            
        case '/help':
            sendHelpMessage($chat_id);
            break;
            
        case '/myorders':
            sendUserOrders($chat_id, $user);
            break;
            
        case '/neworder':
            startNewOrder($chat_id, $user);
            break;
            
        case '/admin':
            handleAdminCommand($chat_id, $user);
            break;
            
        default:
            sendMessage($chat_id, "دستور نامعتبر است. برای مشاهده راهنما، دستور /help را ارسال کنید.");
    }
}

/**
 * پردازش پیام‌های متنی
 * 
 * @param string $text متن پیام
 * @param int $chat_id شناسه چت
 * @param array $user اطلاعات کاربر فرستنده
 */
function processText($text, $chat_id, $user) {
    // بررسی حالت فعلی کاربر (مثلاً در حال ثبت سفارش، ارسال توضیحات، و غیره)
    // می‌توان از یک جدول در دیتابیس برای ذخیره حالت‌های کاربران استفاده کرد
    
    // فعلاً فقط پاسخ ساده می‌دهیم
    sendMessage($chat_id, "پیام شما دریافت شد. برای مشاهده دستورات، /help را ارسال کنید.");
}

/**
 * پردازش عکس‌های ارسالی (مثلاً رسید پرداخت)
 * 
 * @param array $message پیام کامل
 * @param int $chat_id شناسه چت
 * @param array $user اطلاعات کاربر فرستنده
 */
function processPhoto($message, $chat_id, $user) {
    global $admin_id;
    
    // دریافت شناسه فایل بزرگترین نسخه عکس
    $photos = $message['photo'];
    $file_id = end($photos)['file_id'];
    
    // بررسی اینکه آیا کاربر در حال ارسال رسید است
    // این را باید با یک مکانیزم ذخیره وضعیت کاربر پیاده‌سازی کرد
    
    // برای نمونه، فرض می‌کنیم کاربر در حال ارسال رسید پرداخت است
    $caption = $message['caption'] ?? 'بدون توضیحات';
    
    // ارسال عکس به ادمین
    $text = "رسید پرداخت جدید از کاربر:\n";
    $text .= "نام: " . ($user['first_name'] ?? '') . " " . ($user['last_name'] ?? '') . "\n";
    $text .= "یوزرنیم: @" . ($user['username'] ?? 'ندارد') . "\n";
    $text .= "آیدی تلگرام: " . $user['id'] . "\n";
    $text .= "توضیحات: " . $caption;
    
    // ارسال به ادمین
    if (!empty($admin_id)) {
        sendPhoto($admin_id, $file_id, $text);
        
        // تاییدیه به کاربر
        sendMessage($chat_id, "✅ رسید پرداخت شما با موفقیت ارسال شد. پس از بررسی، نتیجه به شما اطلاع داده خواهد شد.");
    } else {
        sendMessage($chat_id, "❌ متأسفانه در حال حاضر امکان بررسی رسید وجود ندارد. لطفاً بعداً مجدداً تلاش کنید.");
    }
}

/**
 * پردازش کالبک‌های دکمه‌های اینلاین
 * 
 * @param string $data داده کالبک
 * @param int $chat_id شناسه چت
 * @param array $user اطلاعات کاربر
 * @param string $callback_id شناسه کالبک
 */
function processCallback($data, $chat_id, $user, $callback_id) {
    // پاسخ به کالبک برای حذف نشانگر بارگذاری
    answerCallbackQuery($callback_id);
    
    // جداسازی داده کالبک (به فرمت action:param1:param2)
    $parts = explode(':', $data);
    $action = $parts[0];
    
    switch ($action) {
        case 'order':
            // پردازش انتخاب نوع سفارش
            $order_type = $parts[1] ?? '';
            processOrderSelection($chat_id, $user, $order_type);
            break;
            
        case 'confirm_order':
            // تایید سفارش نهایی
            confirmOrder($chat_id, $user, $parts[1] ?? '');
            break;
            
        case 'cancel_order':
            // لغو سفارش
            cancelOrder($chat_id);
            break;
            
        case 'admin_approve':
            // تایید سفارش توسط ادمین
            adminApproveOrder($chat_id, $parts[1] ?? 0);
            break;
            
        case 'admin_reject':
            // رد سفارش توسط ادمین
            adminRejectOrder($chat_id, $parts[1] ?? 0);
            break;
            
        default:
            sendMessage($chat_id, "عملیات نامعتبر است.");
    }
}

/**
 * ارسال پیام شروع
 * 
 * @param int $chat_id شناسه چت
 */
function sendStartMessage($chat_id) {
    $app_name = get_setting('app_name', 'V2Ray Manager');
    $app_description = get_setting('app_description', 'سیستم مدیریت سرویس V2Ray');
    
    $text = "🌟 به ربات $app_name خوش آمدید 🌟\n\n";
    $text .= "$app_description\n\n";
    $text .= "از طریق این ربات می‌توانید:\n";
    $text .= "✅ وضعیت حساب خود را مشاهده کنید\n";
    $text .= "✅ سفارش جدید ثبت کنید\n";
    $text .= "✅ اشتراک خود را تمدید کنید\n";
    $text .= "✅ یک حساب تست دریافت کنید\n\n";
    $text .= "برای شروع، از دکمه‌های زیر استفاده کنید:";
    
    $keyboard = [
        [
            ['text' => '👤 وضعیت حساب من', 'callback_data' => 'status'],
            ['text' => '🆕 سفارش جدید', 'callback_data' => 'order:new']
        ],
        [
            ['text' => '🔄 تمدید اشتراک', 'callback_data' => 'order:renewal'],
            ['text' => '📊 آمار مصرف', 'callback_data' => 'stats']
        ],
        [
            ['text' => '🔑 دریافت اکانت تست', 'callback_data' => 'test_account']
        ]
    ];
    
    sendMessageWithKeyboard($chat_id, $text, $keyboard);
}

/**
 * ارسال وضعیت حساب کاربر
 * 
 * @param int $chat_id شناسه چت
 * @param array $user اطلاعات کاربر
 */
function sendStatusMessage($chat_id, $user) {
    // جستجوی کاربر بر اساس آیدی تلگرام
    global $pdo;
    $telegram_id = $user['id'];
    
    $stmt = $pdo->prepare("SELECT * FROM users WHERE telegram_id = ?");
    $stmt->execute([$telegram_id]);
    $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user_data) {
        $text = "❌ شما هنوز در سیستم ثبت نام نکرده‌اید.\n\n";
        $text .= "برای ثبت سفارش و دریافت اکانت، گزینه «سفارش جدید» را انتخاب کنید.";
        
        $keyboard = [
            [
                ['text' => '🆕 سفارش جدید', 'callback_data' => 'order:new']
            ],
            [
                ['text' => '🔑 دریافت اکانت تست', 'callback_data' => 'test_account']
            ]
        ];
        
        sendMessageWithKeyboard($chat_id, $text, $keyboard);
        return;
    }
    
    // محاسبه روزهای باقیمانده
    $expiration = strtotime($user_data['expiration'] ?? '');
    $days_left = $expiration ? ceil(($expiration - time()) / 86400) : 0;
    
    // محاسبه مصرف
    $quota_gb = $user_data['quota'] > 0 ? round($user_data['quota'] / 1073741824, 2) : '∞';
    $usage_gb = round($user_data['usage'] / 1073741824, 2);
    $percentage = $user_data['quota'] > 0 ? round(($user_data['usage'] / $user_data['quota']) * 100, 1) : 0;
    
    $text = "👤 اطلاعات حساب شما\n\n";
    $text .= "🔹 نام کاربری: " . $user_data['username'] . "\n";
    $text .= "🔹 وضعیت: " . ($user_data['is_blocked'] ? "⛔️ مسدود" : "✅ فعال") . "\n";
    $text .= "🔹 تاریخ انقضا: " . ($user_data['expiration'] ? jdate('Y/m/d', $expiration) : '∞') . "\n";
    $text .= "🔹 روزهای باقیمانده: " . ($days_left > 0 ? $days_left : "منقضی شده") . " روز\n";
    $text .= "🔹 حجم کل: " . $quota_gb . " GB\n";
    $text .= "🔹 حجم مصرف شده: " . $usage_gb . " GB (" . $percentage . "%)\n";
    $text .= "🔹 دستگاه‌های فعال: " . $user_data['active_devices'] . " از " . $user_data['max_devices'] . "\n";
    
    $keyboard = [
        [
            ['text' => '🔄 تمدید اشتراک', 'callback_data' => 'order:renewal'],
            ['text' => '📊 آمار مصرف', 'callback_data' => 'stats']
        ],
        [
            ['text' => '📱 دستگاه‌های من', 'callback_data' => 'devices'],
            ['text' => '📋 سفارش‌های من', 'callback_data' => 'my_orders']
        ]
    ];
    
    sendMessageWithKeyboard($chat_id, $text, $keyboard);
}

/**
 * ایجاد حساب تست برای کاربر
 * 
 * @param int $chat_id شناسه چت
 * @param array $user اطلاعات کاربر
 */
function createTestAccount($chat_id, $user) {
    global $pdo;
    $telegram_id = $user['id'];
    
    // بررسی اینکه کاربر قبلاً حساب تست داشته یا خیر
    $stmt = $pdo->prepare("SELECT id FROM users WHERE telegram_id = ? AND is_test_account = 1");
    $stmt->execute([$telegram_id]);
    
    if ($stmt->fetch()) {
        sendMessage($chat_id, "❌ شما قبلاً از حساب تست استفاده کرده‌اید و امکان دریافت مجدد وجود ندارد.");
        return;
    }
    
    // ایجاد نام کاربری تصادفی
    $username = "test_" . substr(md5($telegram_id . time()), 0, 8);
    $password = substr(md5(time() . $telegram_id), 0, 8);
    
    // ایجاد حساب تست
    $user_id = create_user($username, $password, null, $telegram_id, true);
    
    if ($user_id) {
        $test_days = get_setting('test_account_duration', 1);
        $test_quota_gb = get_setting('test_account_quota', 1073741824) / 1073741824;
        
        $text = "✅ حساب تست شما با موفقیت ایجاد شد!\n\n";
        $text .= "🔹 نام کاربری: $username\n";
        $text .= "🔹 رمز عبور: $password\n";
        $text .= "🔹 مدت اعتبار: $test_days روز\n";
        $text .= "🔹 حجم: $test_quota_gb GB\n\n";
        $text .= "برای اتصال، به برنامه V2Ray خود مراجعه کرده و اطلاعات فوق را وارد کنید.";
        
        sendMessage($chat_id, $text);
    } else {
        sendMessage($chat_id, "❌ متأسفانه در ایجاد حساب تست خطایی رخ داد. لطفاً بعداً مجدداً تلاش کنید.");
    }
}

/**
 * شروع فرآیند ثبت سفارش جدید
 * 
 * @param int $chat_id شناسه چت
 * @param array $user اطلاعات کاربر
 */
function startNewOrder($chat_id, $user) {
    $text = "👇 لطفاً نوع سفارش خود را انتخاب کنید:";
    
    $keyboard = [
        [
            ['text' => '🆕 اشتراک جدید', 'callback_data' => 'order:new']
        ],
        [
            ['text' => '🔄 تمدید اشتراک', 'callback_data' => 'order:renewal']
        ],
        [
            ['text' => '⬆️ ارتقاء حساب (افزایش دستگاه)', 'callback_data' => 'order:upgrade']
        ],
        [
            ['text' => '❌ انصراف', 'callback_data' => 'cancel_order']
        ]
    ];
    
    sendMessageWithKeyboard($chat_id, $text, $keyboard);
}

/**
 * ارسال پیام راهنما
 * 
 * @param int $chat_id شناسه چت
 */
function sendHelpMessage($chat_id) {
    $text = "🔰 راهنمای استفاده از ربات:\n\n";
    $text .= "👈 دستورات اصلی:\n";
    $text .= "/start - شروع کار با ربات\n";
    $text .= "/status - مشاهده وضعیت حساب\n";
    $text .= "/test - دریافت اکانت تست\n";
    $text .= "/neworder - ثبت سفارش جدید\n";
    $text .= "/myorders - مشاهده سفارش‌های من\n";
    $text .= "/help - راهنمای ربات\n\n";
    
    $text .= "👈 برای دریافت پشتیبانی، به ادمین پیام دهید: ";
    $text .= "@" . (get_setting('support_username', 'admin'));
    
    sendMessage($chat_id, $text);
}

/**
 * ارسال پیام متنی ساده
 * 
 * @param int $chat_id شناسه چت
 * @param string $text متن پیام
 * @return bool نتیجه ارسال
 */
function sendMessage($chat_id, $text) {
    global $bot_token;
    
    $url = "https://api.telegram.org/bot$bot_token/sendMessage";
    $data = [
        'chat_id' => $chat_id,
        'text' => $text,
        'parse_mode' => 'Markdown'
    ];
    
    $options = [
        'http' => [
            'header' => "Content-type: application/x-www-form-urlencoded\r\n",
            'method' => 'POST',
            'content' => http_build_query($data)
        ]
    ];
    
    $context = stream_context_create($options);
    $result = file_get_contents($url, false, $context);
    
    return $result !== false;
}

/**
 * ارسال پیام با کیبورد اینلاین
 * 
 * @param int $chat_id شناسه چت
 * @param string $text متن پیام
 * @param array $keyboard کیبورد
 * @return bool نتیجه ارسال
 */
function sendMessageWithKeyboard($chat_id, $text, $keyboard) {
    global $bot_token;
    
    $url = "https://api.telegram.org/bot$bot_token/sendMessage";
    $data = [
        'chat_id' => $chat_id,
        'text' => $text,
        'parse_mode' => 'Markdown',
        'reply_markup' => json_encode([
            'inline_keyboard' => $keyboard
        ])
    ];
    
    $options = [
        'http' => [
            'header' => "Content-type: application/x-www-form-urlencoded\r\n",
            'method' => 'POST',
            'content' => http_build_query($data)
        ]
    ];
    
    $context = stream_context_create($options);
    $result = file_get_contents($url, false, $context);
    
    return $result !== false;
}

/**
 * ارسال عکس
 * 
 * @param int $chat_id شناسه چت
 * @param string $photo_id شناسه عکس یا آدرس آن
 * @param string $caption متن توضیحات
 * @return bool نتیجه ارسال
 */
function sendPhoto($chat_id, $photo_id, $caption = '') {
    global $bot_token;
    
    $url = "https://api.telegram.org/bot$bot_token/sendPhoto";
    $data = [
        'chat_id' => $chat_id,
        'photo' => $photo_id,
        'caption' => $caption,
        'parse_mode' => 'Markdown'
    ];
    
    $options = [
        'http' => [
            'header' => "Content-type: application/x-www-form-urlencoded\r\n",
            'method' => 'POST',
            'content' => http_build_query($data)
        ]
    ];
    
    $context = stream_context_create($options);
    $result = file_get_contents($url, false, $context);
    
    return $result !== false;
}

/**
 * پاسخ به کالبک (برای حذف نشانگر بارگذاری)
 * 
 * @param string $callback_id شناسه کالبک
 * @param string $text متن پیام (اختیاری)
 * @param bool $show_alert نمایش به صورت هشدار
 * @return bool نتیجه ارسال
 */
function answerCallbackQuery($callback_id, $text = '', $show_alert = false) {
    global $bot_token;
    
    $url = "https://api.telegram.org/bot$bot_token/answerCallbackQuery";
    $data = [
        'callback_query_id' => $callback_id,
        'text' => $text,
        'show_alert' => $show_alert
    ];
    
    $options = [
        'http' => [
            'header' => "Content-type: application/x-www-form-urlencoded\r\n",
            'method' => 'POST',
            'content' => http_build_query($data)
        ]
    ];
    
    $context = stream_context_create($options);
    $result = file_get_contents($url, false, $context);
    
    return $result !== false;
}

// تابع برای تاریخ شمسی - این را باید پیاده‌سازی کنید یا از کتابخانه استفاده کنید
function jdate($format, $timestamp) {
    // پیاده‌سازی تبدیل تاریخ میلادی به شمسی
    // برای مثال، اینجا فقط تاریخ میلادی را برمی‌گردانیم
    return date($format, $timestamp);
}

// سایر توابع مورد نیاز برای پردازش سفارش‌ها، تایید ادمین، و غیره...
?> 